/********************************************************************
 FileName       : main.c
 Dependencies   : See INCLUDES section
 Processor      : PIC18, PIC24, and PIC32 USB Microcontrollers
 Hardware       : This demo is natively intended to be used on Microchip USB demo
 		  boards supported by the MCHPFSUSB stack.  See release notes for
 		  support matrix.  This demo can be modified for use on other hardware
 		  platforms.
 Complier       : Microchip C18 (for PIC18), C30 (for PIC24), C32 (for PIC32)
 Company        : Microchip Technology, Inc.

 Software License Agreement:

 The software supplied herewith by Microchip Technology Incorporated
 (the “Company”) for its PIC® Microcontroller is intended and
 supplied to you, the Company’s customer, for use solely and
 exclusively on Microchip PIC Microcontroller products. The
 software is owned by the Company and/or its supplier, and is
 protected under applicable copyright laws. All rights are reserved.
 Any use in violation of the foregoing restrictions may subject the
 user to criminal sanctions under applicable laws, as well as to
 civil liability for the breach of the terms and conditions of this
 license.

 THIS SOFTWARE IS PROVIDED IN AN “AS IS” CONDITION. NO WARRANTIES,
 WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
 TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,
 IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
 CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.

********************************************************************
 File Description:

 Change History:
  Rev   Description
  1.0   Initial release
  2.1   Updated for simplicity and to use common
                     coding style
********************************************************************/
#include <plib.h>           /* Include to use PIC32 peripheral libraries      */

#include "USB/usb.h"
#include "USB/usb_host_msd.h"
#include "USB/usb_host_msd_scsi.h"
#include "MDD File System/FSIO.h"

// *****************************************************************************
// *****************************************************************************
// Configuration Bits
// *****************************************************************************
// *****************************************************************************

#if defined(ET_BASE_PIC32MX)
    #include "Configuration base-pic32mx.h"

//ET-BASE PIC32(PIC32MX250F128B)
#elif defined( __PIC32MX__ )
  #pragma config UPLLEN   = ON            // USB PLL Enabled
  #pragma config FPLLMUL  = MUL_15        // PLL Multiplier
  #pragma config UPLLIDIV = DIV_2         // USB PLL Input Divider
  #pragma config FPLLIDIV = DIV_2         // PLL Input Divider
  #pragma config FPLLODIV = DIV_1         // PLL Output Divider
  #pragma config FPBDIV   = DIV_1         // Peripheral Clock divisor
  #pragma config FWDTEN   = OFF           // Watchdog Timer
  #pragma config WDTPS    = PS1           // Watchdog Timer Postscale
    //#pragma config FCKSM    = CSDCMD        // Clock Switching & Fail Safe Clock Monitor
  #pragma config OSCIOFNC = OFF           // CLKO Enable
  #pragma config POSCMOD  = HS            // Primary Oscillator
  #pragma config IESO     = OFF           // Internal/External Switch-over
  #pragma config FSOSCEN  = OFF           // Secondary Oscillator Enable (KLO was off)
  #pragma config FNOSC    = PRIPLL        // Oscillator Selection
  #pragma config CP       = OFF           // Code Protect
  #pragma config BWP      = OFF           // Boot Flash Write Protect
  #pragma config PWP      = OFF           // Program Flash Write Protect
  #pragma config ICESEL   = ICS_PGx1      // ICE/ICD Comm Channel Select
#else
    #error Cannot define configuration bits.
#endif

FSFILE * myFile;
BYTE myDataWrite[512];
size_t numBytes;
volatile BOOL deviceAttached;
char myDataRead[50];


/******************************************************************************/
/* Global Variable Declaration                                                */
/******************************************************************************/
void delay(unsigned long delay_count);

void SendDataBufferUART1(const char *buffer, UINT32 size);
UINT32 GetDataBufferUART1(char *buffer, UINT32 max_size);
UINT32 GetMenuChoiceUART1(void);

void SendDataBufferUART2(const char *buffer, UINT32 size);
UINT32 GetDataBufferUART2(char *buffer, UINT32 max_size);
UINT32 GetMenuChoiceUART2(void);

const char mainMenuUART1[] =
{
  "\r\n\r\n"\
  "=======================================\r\n"\
  "ET-BASE PIC32 : PIC32MX250F128B UART[1]\r\n"\
  "Run External Crystal(8MHz)+PLL = 48MHz\r\n"\
  "-> Test GPIO RA1\r\n"\
  "-> Test GPIO RB0,1,2,3,4,5,7\r\n"\
  "-> Test GPIO RB13,14,15\r\n"\
  "======================================= \r\n"\
  "Press any key For Echo Test UART1...\r\n"\
  "Press 'Q'...For Quit Test UART1\r\n"\
  "======================================= \r\n"\
  "UART1>"
};

// Constant Data
const char mainMenuUART2[] =
{
  "\r\n\r\n"\
  "=======================================\r\n"\
  "ET-BASE PIC32 : PIC32MX250F128B UART[2]\r\n"\
  "Run External Crystal(8MHz)+PLL = 48MHz\r\n"\
  "-> Test GPIO RA1\r\n"\
  "-> Test GPIO RB0,1,2,3,4,5,7\r\n"\
  "-> Test GPIO RB13,14,15\r\n"\
  "======================================= \r\n"\
  "Press any key For Echo Test UART2...\r\n"\
  "Press 'Q'...For Quit Test UART2\r\n"\
  "======================================= \r\n"\
  "UART2>"
};

//==============================================================================
int main(void)
{
  unsigned long blink_counter;
  UINT32  rx_data;
  unsigned long i;
  BYTE led_count;
  BOOL StopTest;

  //ET-BASE PIC32(PIC32MX250F128B)
  #if defined(__PIC32MX__)
  {
    int  value;
    
    value = SYSTEMConfigWaitStatesAndPB( GetSystemClock() );
    
    // Enable the cache for the best performance
    CheKseg0CacheOn();
    
    INTEnableSystemMultiVectoredInt();
    
    value = OSCCON;
    while (!(value & 0x00000020))
    {
      value = OSCCON;    // Wait for PLL lock to stabilize
    }    
  }
  #endif

  deviceAttached = FALSE;

  //Initialize the stack
  USBInitialize(0);

  //============================================================================
  //Initial All GPIO = Output
  //PORTSetPinsDigitalOut(IOPORT_A, BIT_0);     // TX1
  PORTSetPinsDigitalOut(IOPORT_A, BIT_1);
  //PORTSetPinsDigitalOut(IOPORT_A, BIT_2);     // CKI
  //PORTSetPinsDigitalOut(IOPORT_A, BIT_3);     // CKO
  //PORTSetPinsDigitalOut(IOPORT_A, BIT_4);     // RX1

  PORTSetPinsDigitalOut(IOPORT_B, BIT_0);
  PORTSetPinsDigitalOut(IOPORT_B, BIT_1);
  PORTSetPinsDigitalOut(IOPORT_B, BIT_2);
  PORTSetPinsDigitalOut(IOPORT_B, BIT_3);
  PORTSetPinsDigitalOut(IOPORT_B, BIT_4);
  PORTSetPinsDigitalOut(IOPORT_B, BIT_5);
  //PORTSetPinsDigitalOut(IOPORT_B, BIT_6);     // VBUS
  PORTSetPinsDigitalOut(IOPORT_B, BIT_7);
  //PORTSetPinsDigitalOut(IOPORT_B, BIT_8);     // RX2
  //PORTSetPinsDigitalOut(IOPORT_B, BIT_9);     // TX2
  //PORTSetPinsDigitalOut(IOPORT_B, BIT_10);    // D-
  //PORTSetPinsDigitalOut(IOPORT_B, BIT_11);    // D+
  //PORTSetPinsDigitalOut(IOPORT_B, BIT_12);    // VUSB(3V3)
  PORTSetPinsDigitalOut(IOPORT_B, BIT_13);
  PORTSetPinsDigitalOut(IOPORT_B, BIT_14);
  PORTSetPinsDigitalOut(IOPORT_B, BIT_15);

  //Mapping RA4=RX1,RA0=TX1
  PPSInput(3,U1RX,RPA4);                                                        // Assign RPA4 as input pin for U1RX
  PPSOutput(1,RPA0,U1TX);                                                       // Set RPA0 pin as output for U1TX

  //Mapping RB8=RX2,RB9=TX2
  PPSInput(2,U2RX,RPB8);                                                        // Assign RPB8 as input pin for U2RX
  PPSOutput(4,RPB9,U2TX);                                                       // Set RPB9 pin as output for U2TX

  //Initial UART1
  UARTConfigure(UART1, UART_ENABLE_PINS_TX_RX_ONLY);
  UARTSetFifoMode(UART1, UART_INTERRUPT_ON_TX_NOT_FULL | UART_INTERRUPT_ON_RX_NOT_EMPTY);
  UARTSetLineControl(UART1, UART_DATA_SIZE_8_BITS | UART_PARITY_NONE | UART_STOP_BITS_1);
  UARTSetDataRate(UART1, GetPeripheralClock(), 9600);
  UARTEnable(UART1, UART_ENABLE_FLAGS(UART_PERIPHERAL | UART_RX | UART_TX));

  //Initial UART2
  UARTConfigure(UART2, UART_ENABLE_PINS_TX_RX_ONLY);
  UARTSetFifoMode(UART2, UART_INTERRUPT_ON_TX_NOT_FULL | UART_INTERRUPT_ON_RX_NOT_EMPTY);
  UARTSetLineControl(UART2, UART_DATA_SIZE_8_BITS | UART_PARITY_NONE | UART_STOP_BITS_1);
  UARTSetDataRate(UART2, GetPeripheralClock(), 9600);
  UARTEnable(UART2, UART_ENABLE_FLAGS(UART_PERIPHERAL | UART_RX | UART_TX));

  //Display Menu On UART
  SendDataBufferUART1(mainMenuUART1, sizeof(mainMenuUART1));
  SendDataBufferUART2(mainMenuUART2, sizeof(mainMenuUART2));

  //============================================================================
  StopTest = FALSE;
  led_count = 0;
  while(1)
  {
    for(i=0; i<(SYS_FREQ/100); i++)
    {
      //UART1
      if(UARTReceivedDataIsAvailable(UART1))					// If RX Receive Complete
      {
        rx_data = UARTGetDataByte(UART1);				        // Get UART1 Data Receive
        if(rx_data == 0x0D)							// If Receive Enter
        {
          SendDataBufferUART1("\r\nUART1>",9);
        }
        else if((rx_data == 'Q')||(rx_data == 'q'))                             // If Receive 'Q' = Quit
        {
          SendDataBufferUART1("\r\nStop Test GPIO...",20);
          StopTest = TRUE;
        }
        else
        {
          while(!UARTTransmitterIsReady(UART1));
          UARTSendDataByte(UART1, rx_data);
        }
        while(!UARTTransmissionHasCompleted(UART1));
      }

      //UART2
      if(UARTReceivedDataIsAvailable(UART2))					// If RX Receive Complete
      {
        rx_data = UARTGetDataByte(UART2);				        // Get UART1 Data Receive
        if(rx_data == 0x0D)							// If Receive Enter
        {
          SendDataBufferUART2("\r\nUART2>",9);
        }
        else if((rx_data == 'Q')||(rx_data == 'q'))                             // If Receive 'Q' = Quit
        {
          SendDataBufferUART2("\r\nStop Test GPIO...",20);
          StopTest = TRUE;
        }
        
        else
        {
          while(!UARTTransmitterIsReady(UART2));
          UARTSendDataByte(UART2, rx_data);
        }
        while(!UARTTransmissionHasCompleted(UART2));
      }
    }

    //Blink LED    
    led_count++;
    switch(led_count & 0x07)                            //00000111
    {
      case 0:  //PORTSetBits(IOPORT_A, BIT_0);          // TX1
               PORTSetBits(IOPORT_A, BIT_1);
               //PORTSetBits(IOPORT_A, BIT_2);          // CKI
               //PORTSetBits(IOPORT_A, BIT_3);          // CKO
               //PORTSetBits(IOPORT_A, BIT_4);          // RX1
               PORTSetBits(IOPORT_B, BIT_0);
               PORTSetBits(IOPORT_B, BIT_1);
               PORTSetBits(IOPORT_B, BIT_2);
               PORTSetBits(IOPORT_B, BIT_3);
               PORTSetBits(IOPORT_B, BIT_4);
               PORTSetBits(IOPORT_B, BIT_5);
               //PORTSetBits(IOPORT_B, BIT_6);          // Reserve For : +VBUS
               PORTSetBits(IOPORT_B, BIT_7);
               //PORTSetBits(IOPORT_B, BIT_8);          // RX2
               //PORTSetBits(IOPORT_B, BIT_9);          // TX2
               //PORTSetBits(IOPORT_B, BIT_10);         // D(-)
               //PORTSetBits(IOPORT_B, BIT_11);         // D(+)
               //PORTSetBits(IOPORT_B, BIT_12);         // Reserve For : +VUSB(3V3)
               PORTSetBits(IOPORT_B, BIT_13);
               PORTSetBits(IOPORT_B, BIT_14);
               PORTSetBits(IOPORT_B, BIT_15);
               //
               //ON GPIO:LED
               //mPORTAClearBits(BIT_0);                // RA0(TX1)
               mPORTBClearBits(BIT_0);                  // RB0 -> Reserve For : PGED1
               //mPORTBClearBits(BIT_8);                // RB8(RX2)
               break;

      case 1:  //PORTSetBits(IOPORT_A, BIT_0);          // TX1
               PORTSetBits(IOPORT_A, BIT_1);
               //PORTSetBits(IOPORT_A, BIT_2);          // CKI
               //PORTSetBits(IOPORT_A, BIT_3);          // CKO
               //PORTSetBits(IOPORT_A, BIT_4);          // RX1
               PORTSetBits(IOPORT_B, BIT_0);
               PORTSetBits(IOPORT_B, BIT_1);
               PORTSetBits(IOPORT_B, BIT_2);
               PORTSetBits(IOPORT_B, BIT_3);
               PORTSetBits(IOPORT_B, BIT_4);
               PORTSetBits(IOPORT_B, BIT_5);
               //PORTSetBits(IOPORT_B, BIT_6);          // Reserve For : +VBUS
               PORTSetBits(IOPORT_B, BIT_7);
               //PORTSetBits(IOPORT_B, BIT_8);          // RX2
               //PORTSetBits(IOPORT_B, BIT_9);          // TX2
               //PORTSetBits(IOPORT_B, BIT_10);         // D(-)
               //PORTSetBits(IOPORT_B, BIT_11);         // D(+)
               //PORTSetBits(IOPORT_B, BIT_12);         // Reserve For : +VUSB(3V3)
               PORTSetBits(IOPORT_B, BIT_13);
               PORTSetBits(IOPORT_B, BIT_14);
               PORTSetBits(IOPORT_B, BIT_15);
               //
               //ON GPIO:LED
               mPORTAClearBits(BIT_1);                  // RA1
               mPORTBClearBits(BIT_1);                  // RB1 -> Reserve For : PGEC1
               //mPORTBClearBits(BIT_9);                // RB9(TX2)
               break;

      case 2:  //PORTSetBits(IOPORT_A, BIT_0);          // TX1
               PORTSetBits(IOPORT_A, BIT_1);
               //PORTSetBits(IOPORT_A, BIT_2);          // CKI
               //PORTSetBits(IOPORT_A, BIT_3);          // CKO
               //PORTSetBits(IOPORT_A, BIT_4);          // RX1
               PORTSetBits(IOPORT_B, BIT_0);
               PORTSetBits(IOPORT_B, BIT_1);
               PORTSetBits(IOPORT_B, BIT_2);
               PORTSetBits(IOPORT_B, BIT_3);
               PORTSetBits(IOPORT_B, BIT_4);
               PORTSetBits(IOPORT_B, BIT_5);
               //PORTSetBits(IOPORT_B, BIT_6);          // Reserve For : +VBUS
               PORTSetBits(IOPORT_B, BIT_7);
               //PORTSetBits(IOPORT_B, BIT_8);          // RX2
               //PORTSetBits(IOPORT_B, BIT_9);          // TX2
               //PORTSetBits(IOPORT_B, BIT_10);         // D(-)
               //PORTSetBits(IOPORT_B, BIT_11);         // D(+)
               //PORTSetBits(IOPORT_B, BIT_12);         // Reserve For : +VUSB(3V3)
               PORTSetBits(IOPORT_B, BIT_13);
               PORTSetBits(IOPORT_B, BIT_14);
               PORTSetBits(IOPORT_B, BIT_15);
               //
               //ON GPIO:LED
               //mPORTAClearBits(BIT_2);                // RA2(CKI)
               mPORTBClearBits(BIT_2);                  // RB2
               //mPORTBClearBits(BIT_10);               // RB10(D-)
               break;

      case 3:  //PORTSetBits(IOPORT_A, BIT_0);          // TX1
               PORTSetBits(IOPORT_A, BIT_1);
               //PORTSetBits(IOPORT_A, BIT_2);          // CKI
               //PORTSetBits(IOPORT_A, BIT_3);          // CKO
               //PORTSetBits(IOPORT_A, BIT_4);          // RX1
               PORTSetBits(IOPORT_B, BIT_0);
               PORTSetBits(IOPORT_B, BIT_1);
               PORTSetBits(IOPORT_B, BIT_2);
               PORTSetBits(IOPORT_B, BIT_3);
               PORTSetBits(IOPORT_B, BIT_4);
               PORTSetBits(IOPORT_B, BIT_5);
               //PORTSetBits(IOPORT_B, BIT_6);          // Reserve For : +VBUS
               PORTSetBits(IOPORT_B, BIT_7);
               //PORTSetBits(IOPORT_B, BIT_8);          // RX2
               //PORTSetBits(IOPORT_B, BIT_9);          // TX2
               //PORTSetBits(IOPORT_B, BIT_10);         // D(-)
               //PORTSetBits(IOPORT_B, BIT_11);         // D(+)
               //PORTSetBits(IOPORT_B, BIT_12);         // Reserve For : +VUSB(3V3)
               PORTSetBits(IOPORT_B, BIT_13);
               PORTSetBits(IOPORT_B, BIT_14);
               PORTSetBits(IOPORT_B, BIT_15);
               //
               //ON GPIO:LED
               //mPORTAClearBits(BIT_3);                // RA3(CKO)
               mPORTBClearBits(BIT_3);                  // RB3
               //mPORTBClearBits(BIT_11);               // RB11(D+)
               break;

      case 4:  //PORTSetBits(IOPORT_A, BIT_0);          // TX1
               PORTSetBits(IOPORT_A, BIT_1);
               //PORTSetBits(IOPORT_A, BIT_2);          // CKI
               //PORTSetBits(IOPORT_A, BIT_3);          // CKO
               //PORTSetBits(IOPORT_A, BIT_4);          // RX1
               PORTSetBits(IOPORT_B, BIT_0);
               PORTSetBits(IOPORT_B, BIT_1);
               PORTSetBits(IOPORT_B, BIT_2);
               PORTSetBits(IOPORT_B, BIT_3);
               PORTSetBits(IOPORT_B, BIT_4);
               PORTSetBits(IOPORT_B, BIT_5);
               //PORTSetBits(IOPORT_B, BIT_6);          // Reserve For : +VBUS
               PORTSetBits(IOPORT_B, BIT_7);
               //PORTSetBits(IOPORT_B, BIT_8);          // RX2
               //PORTSetBits(IOPORT_B, BIT_9);          // TX2
               //PORTSetBits(IOPORT_B, BIT_10);         // D(-)
               //PORTSetBits(IOPORT_B, BIT_11);         // D(+)
               //PORTSetBits(IOPORT_B, BIT_12);         // Reserve For : +VUSB(3V3)
               PORTSetBits(IOPORT_B, BIT_13);
               PORTSetBits(IOPORT_B, BIT_14);
               PORTSetBits(IOPORT_B, BIT_15);
               //
               //ON GPIO:LED
               //mPORTAClearBits(BIT_4);                // RA4(RX1)
               mPORTBClearBits(BIT_4);                  // RB4
               //mPORTBClearBits(BIT_12);               // Reserve For : +VUSB(3V3)
               break;

      case 5:  //PORTSetBits(IOPORT_A, BIT_0);          // TX1
               PORTSetBits(IOPORT_A, BIT_1);
               //PORTSetBits(IOPORT_A, BIT_2);          // CKI
               //PORTSetBits(IOPORT_A, BIT_3);          // CKO
               //PORTSetBits(IOPORT_A, BIT_4);          // RX1
               PORTSetBits(IOPORT_B, BIT_0);
               PORTSetBits(IOPORT_B, BIT_1);
               PORTSetBits(IOPORT_B, BIT_2);
               PORTSetBits(IOPORT_B, BIT_3);
               PORTSetBits(IOPORT_B, BIT_4);
               PORTSetBits(IOPORT_B, BIT_5);
               //PORTSetBits(IOPORT_B, BIT_6);          // Reserve For : +VBUS
               PORTSetBits(IOPORT_B, BIT_7);
               //PORTSetBits(IOPORT_B, BIT_8);          // RX2
               //PORTSetBits(IOPORT_B, BIT_9);          // TX2
               //PORTSetBits(IOPORT_B, BIT_10);         // D(-)
               //PORTSetBits(IOPORT_B, BIT_11);         // D(+)
               //PORTSetBits(IOPORT_B, BIT_12);         // Reserve For : +VUSB(3V3)
               PORTSetBits(IOPORT_B, BIT_13);
               PORTSetBits(IOPORT_B, BIT_14);
               PORTSetBits(IOPORT_B, BIT_15);
               //
               //ON GPIO:LED
               mPORTBClearBits(BIT_5);                   // RB5
               mPORTBClearBits(BIT_13);                  // RB13
               break;

      case 6:  //PORTSetBits(IOPORT_A, BIT_0);          // TX1
               PORTSetBits(IOPORT_A, BIT_1);
               //PORTSetBits(IOPORT_A, BIT_2);          // CKI
               //PORTSetBits(IOPORT_A, BIT_3);          // CKO
               //PORTSetBits(IOPORT_A, BIT_4);          // RX1
               PORTSetBits(IOPORT_B, BIT_0);
               PORTSetBits(IOPORT_B, BIT_1);
               PORTSetBits(IOPORT_B, BIT_2);
               PORTSetBits(IOPORT_B, BIT_3);
               PORTSetBits(IOPORT_B, BIT_4);
               PORTSetBits(IOPORT_B, BIT_5);
               //PORTSetBits(IOPORT_B, BIT_6);          // Reserve For : +VBUS
               PORTSetBits(IOPORT_B, BIT_7);
               //PORTSetBits(IOPORT_B, BIT_8);          // RX2
               //PORTSetBits(IOPORT_B, BIT_9);          // TX2
               //PORTSetBits(IOPORT_B, BIT_10);         // D(-)
               //PORTSetBits(IOPORT_B, BIT_11);         // D(+)
               //PORTSetBits(IOPORT_B, BIT_12);         // Reserve For : +VUSB(3V3)
               PORTSetBits(IOPORT_B, BIT_13);
               PORTSetBits(IOPORT_B, BIT_14);
               PORTSetBits(IOPORT_B, BIT_15);
               //
               //ON GPIO:LED
               //mPORTBClearBits(BIT_6);                // Reserve For : +VBUS
               mPORTBClearBits(BIT_14);                 // RB14
               break;

      case 7:  //PORTSetBits(IOPORT_A, BIT_0);          // TX1
               PORTSetBits(IOPORT_A, BIT_1);
               //PORTSetBits(IOPORT_A, BIT_2);          // CKI
               //PORTSetBits(IOPORT_A, BIT_3);          // CKO
               //PORTSetBits(IOPORT_A, BIT_4);          // RX1
               PORTSetBits(IOPORT_B, BIT_0);
               PORTSetBits(IOPORT_B, BIT_1);
               PORTSetBits(IOPORT_B, BIT_2);
               PORTSetBits(IOPORT_B, BIT_3);
               PORTSetBits(IOPORT_B, BIT_4);
               PORTSetBits(IOPORT_B, BIT_5);
               //PORTSetBits(IOPORT_B, BIT_6);          // Reserve For : +VBUS
               PORTSetBits(IOPORT_B, BIT_7);
               //PORTSetBits(IOPORT_B, BIT_8);          // RX2
               //PORTSetBits(IOPORT_B, BIT_9);          // TX2
               //PORTSetBits(IOPORT_B, BIT_10);         // D(-)
               //PORTSetBits(IOPORT_B, BIT_11);         // D(+)
               //PORTSetBits(IOPORT_B, BIT_12);         // Reserve For : +VUSB(3V3)
               PORTSetBits(IOPORT_B, BIT_13);
               PORTSetBits(IOPORT_B, BIT_14);
               PORTSetBits(IOPORT_B, BIT_15);
               //
               //ON GPIO:LED
               mPORTBClearBits(BIT_7);                  // RB7
               mPORTBClearBits(BIT_15);                 // RB15
               break;
    }

    // Verify 'Q' : Quit Test
    if(StopTest == TRUE)
    {
      //PORTSetBits(IOPORT_A, BIT_0);                   // TX1
      PORTSetBits(IOPORT_A, BIT_1);
      //PORTSetBits(IOPORT_A, BIT_2);                   // CKI
      //PORTSetBits(IOPORT_A, BIT_3);                   // CKO
      //PORTSetBits(IOPORT_A, BIT_4);                   // RX1
      PORTSetBits(IOPORT_B, BIT_0);
      PORTSetBits(IOPORT_B, BIT_1);
      PORTSetBits(IOPORT_B, BIT_2);
      PORTSetBits(IOPORT_B, BIT_3);
      PORTSetBits(IOPORT_B, BIT_4);
      PORTSetBits(IOPORT_B, BIT_5);
      //PORTSetBits(IOPORT_B, BIT_6);                   // Reserve For : +VBUS
      PORTSetBits(IOPORT_B, BIT_7);
      //PORTSetBits(IOPORT_B, BIT_8);                   // RX2
      //PORTSetBits(IOPORT_B, BIT_9);                   // TX2
      //PORTSetBits(IOPORT_B, BIT_10);                  // D(-)
      //PORTSetBits(IOPORT_B, BIT_11);                  // D(+)
      //PORTSetBits(IOPORT_B, BIT_12);                   // Reserve For : +VUSB(3V3)
      PORTSetBits(IOPORT_B, BIT_13);
      PORTSetBits(IOPORT_B, BIT_14);
      PORTSetBits(IOPORT_B, BIT_15);
      //
      break;
    }
  }
  //============================================================================
  SendDataBufferUART1("\r\nStart Test USB Host MSD...",29);
  SendDataBufferUART2("\r\nStart Test USB Host MSD...",29);

  //Initialize all of the LED pins
  mInitAllLEDs();

  //Start Operate Status
  mLED_1_On(); mLED_2_On();
  blink_counter = 0;
  
  while(1)
  {
    //USB stack process function
    USBTasks();

    blink_counter++; 
    if(blink_counter>(SYS_FREQ/250))
    { 
      blink_counter = 0;
      mLED_1_Toggle();
      mLED_2_Toggle();
    }

    //if thumbdrive is plugged in
    if(USBHostMSDSCSIMediaDetect())
    {
      deviceAttached = TRUE;

      //Connect USB MSD Device Status
      mLED_1_On(); mLED_2_On();

      //now a device is attached
      //See if the device is attached and in the right format
      if(FSInit())
      {
        // Start of Initial USB MSD Device Status
        mLED_1_On(); mLED_2_Off();

        SendDataBufferUART1("\r\nStart Test Write USB Host MSD...",35);
        SendDataBufferUART2("\r\nStart Test Write USB Host MSD...",35);

        // Opening a file in mode "w" will create the file if it doesn't
        // exist.  If the file does exist it will delete the old file
        // and create a new one that is blank.
        myFile = FSfopen("test.txt","w");

        // Write some data to the new file.
        FSfwrite("This is a test.",1,15,myFile);
                
        // Always make sure to close the file so that the data gets
        // written to the drive.
        FSfclose(myFile);

        //======================================================================
        SendDataBufferUART1("\r\nStart Test Read USB Host MSD....",35);
        SendDataBufferUART2("\r\nStart Test Read USB Host MSD....",35);

        // Open file in read mode
        myFile = FSfopen ("test.txt", "r");
        if(myFile == NULL)
        while(1);

        // Read 15 Byte
        if(FSfread(myDataRead, 15, 1, myFile) != 1)
        while(1);

        // Check if this is the end of the file- it shouldn't be
        //if(FSfeof(myFile))
        //while(1);

        // Close the file
        if(FSfclose(myFile))
        while(1);

        SendDataBufferUART1("\r\nUSB Host MSD Data = ",23);
        SendDataBufferUART1(myDataRead,16);

        SendDataBufferUART2("\r\nUSB Host MSD Data = ",23);
        SendDataBufferUART2(myDataRead,16);
        //======================================================================

        // End of Write Text File to USB MSD Device Status
        mLED_1_Off(); mLED_2_On();

        // Just sit here until the device is removed.
        blink_counter = 0;
        while(deviceAttached == TRUE)
        {
          USBTasks();

          blink_counter++;
          if(blink_counter>(SYS_FREQ/250))
          {
            blink_counter = 0;
            mLED_1_Toggle();
            mLED_2_Toggle();
          }
        }

        // Disconnect USB MSD Device Status
        mLED_1_Off(); mLED_2_Off();
      }
    }
  }
  return 0;
}

/****************************************************************************
  Function:
    BOOL USB_ApplicationEventHandler( BYTE address, USB_EVENT event,
                void *data, DWORD size )

  Summary:
    This is the application event handler.  It is called when the stack has
    an event that needs to be handled by the application layer rather than
    by the client driver.

  Description:
    This is the application event handler.  It is called when the stack has
    an event that needs to be handled by the application layer rather than
    by the client driver.  If the application is able to handle the event, it
    returns TRUE.  Otherwise, it returns FALSE.

  Precondition:
    None

  Parameters:
    BYTE address    - Address of device where event occurred
    USB_EVENT event - Identifies the event that occured
    void *data      - Pointer to event-specific data
    DWORD size      - Size of the event-specific data

  Return Values:
    TRUE    - The event was handled
    FALSE   - The event was not handled

  Remarks:
    The application may also implement an event handling routine if it
    requires knowledge of events.  To do so, it must implement a routine that
    matches this function signature and define the USB_HOST_APP_EVENT_HANDLER
    macro as the name of that function.
  ***************************************************************************/

BOOL USB_ApplicationEventHandler( BYTE address, USB_EVENT event, void *data, DWORD size )
{
  switch( event )
  {
    case EVENT_VBUS_REQUEST_POWER:
      // The data pointer points to a byte that represents the amount of power
      // requested in mA, divided by two.  If the device wants too much power,
      // we reject it.
    return TRUE;

    case EVENT_VBUS_RELEASE_POWER:
      // Turn off Vbus power.
      // The PIC24F with the Explorer 16 cannot turn off Vbus through software.

      //This means that the device was removed
      deviceAttached = FALSE;
      return TRUE;
    break;

    case EVENT_HUB_ATTACH:
      return TRUE;
    break;

    case EVENT_UNSUPPORTED_DEVICE:
      return TRUE;
    break;

    case EVENT_CANNOT_ENUMERATE:
      //UART2PrintString( "\r\n***** USB Error - cannot enumerate device *****\r\n" );
      return TRUE;
    break;

    case EVENT_CLIENT_INIT_ERROR:
      //UART2PrintString( "\r\n***** USB Error - client driver initialization error *****\r\n" );
      return TRUE;
    break;

    case EVENT_OUT_OF_MEMORY:
      //UART2PrintString( "\r\n***** USB Error - out of heap memory *****\r\n" );
      return TRUE;
    break;

    case EVENT_UNSPECIFIED_ERROR:   // This should never be generated.
      //UART2PrintString( "\r\n***** USB Error - unspecified *****\r\n" );
      return TRUE;
    break;

    default:
    break;
  }
  return FALSE;
}

/**************************************/
/* Polling Loop Delay 1...delay_count */
/**************************************/
void delay(unsigned long delay_count)
{
  unsigned long i;
  for(i=0; i<delay_count; i++);
}

// *****************************************************************************
// void UARTTxBuffer(char *buffer, UINT32 size)
// *****************************************************************************
void SendDataBufferUART1(const char *buffer, UINT32 size)
{
  while(size-1)
  {
    while(!UARTTransmitterIsReady(UART1));
    UARTSendDataByte(UART1, *buffer);
    buffer++;
    size--;
  }
  while(!UARTTransmissionHasCompleted(UART1));
}

// *****************************************************************************
// UINT32 GetDataBuffer(char *buffer, UINT32 max_size)
// *****************************************************************************
UINT32 GetDataBufferUART1(char *buffer, UINT32 max_size)
{
  UINT32 num_char;
  num_char = 0;
  while(num_char < max_size)
  {
    UINT8 character;
    while(!UARTReceivedDataIsAvailable(UART1));
    character = UARTGetDataByte(UART1);

    if(character == '\r')
    break;

    *buffer = character;

    buffer++;
    num_char++;
  }
  return num_char;
}

// *****************************************************************************
// UINT32 GetMenuChoice(void)
// *****************************************************************************
UINT32 GetMenuChoiceUART1(void)
{
  UINT8  menu_item;

  while(!UARTReceivedDataIsAvailable(UART1));
  menu_item = UARTGetDataByte(UART1);
  menu_item -= '0';
  return (UINT32)menu_item;
}

// *****************************************************************************
// void UARTTxBuffer(char *buffer, UINT32 size)
// *****************************************************************************
void SendDataBufferUART2(const char *buffer, UINT32 size)
{
  while(size-1)
  {
    while(!UARTTransmitterIsReady(UART2));
    UARTSendDataByte(UART2, *buffer);
    buffer++;
    size--;
  }
  while(!UARTTransmissionHasCompleted(UART2));
}

// *****************************************************************************
// UINT32 GetDataBuffer(char *buffer, UINT32 max_size)
// *****************************************************************************
UINT32 GetDataBufferUART2(char *buffer, UINT32 max_size)
{
  UINT32 num_char;

  num_char = 0;
  while(num_char < max_size)
  {
    UINT8 character;
    while(!UARTReceivedDataIsAvailable(UART2));
    character = UARTGetDataByte(UART2);

    if(character == '\r')
    break;

    *buffer = character;

    buffer++;
    num_char++;
  }
  return num_char;
}

// *****************************************************************************
// UINT32 GetMenuChoice(void)
// *****************************************************************************
UINT32 GetMenuChoiceUART2(void)
{
  UINT8  menu_item;

  while(!UARTReceivedDataIsAvailable(UART2));
  menu_item = UARTGetDataByte(UART2);
  menu_item -= '0';
  return (UINT32)menu_item;
}
